<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use App\Models\Contact;
use App\Models\Event;
use App\Models\Hotel;
use App\Models\HotelBooking;
use App\Models\NewsLatter;
use App\Models\Package;
use App\Models\Place;
use App\Models\User;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Validator;
use Yajra\DataTables\Facades\DataTables;

class AdminController extends Controller
{

    public function __construct()
    {
        $this->middleware(['auth']);
        Parent::__construct();
    }

    public function destoryImage(Request $request)
    {
        try {
            $request->model::where(['id' => $request->id])->update([
                $request->image_name => ''
            ]);
            destroryFileByFullPath($request->full_path);

            return $this->backWithSuccess(__('Image delete successfully'));
        } catch (\Throwable $th) {
            return $this->backWithError(__('Something wrong. Please try again'));
        }
    }

    public function index()
    {
        try {
            $Hotel_bookiing = HotelBooking::select('id', 'created_at')
                ->where('status', BOOKING_STATUS_CONFIRM)
                ->get()
                ->groupBy(function ($date) {
                    return Carbon::parse($date->created_at)->format('m'); // grouping by months
                });
            $bookingcount = [];
            $bookingMlist = [];
            foreach ($Hotel_bookiing as $key => $value) {
                $bookingcount[(int)$key] = count($value);
            }
            for ($i = 1; $i <= 12; $i++) {
                if (!empty($bookingcount[$i])) {
                    $bookingMlist[$i] = $bookingcount[$i];
                } else {
                    $bookingMlist[$i] = 0;
                }
            }
            $data['bookingMlist'] = $bookingMlist;

            $title = 'My Dashboard';
            $packageCount = Package::count();
            $hotelCount = Hotel::count();
            $placeCount = Place::count();
            $eventCount = Event::count();
            return view('backend.pages.dashboard', compact('title', 'packageCount', 'hotelCount', 'placeCount','eventCount', 'data'));
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function profile()
    {
        try {
            $title = 'My profile';
            return view('backend.pages.profile.show', compact('title'));
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function profileUpdate(Request $request)
    {
        try {
            $input = $request->all();
            if ($request->hasFile('photo')) {
                $input['photo'] = $request->photo[0];
            }
            $user = User::findorfail(Auth::id());


            Validator::make($input, [
                'name' => ['required', 'string', 'max:255'],
                'photo' => ['nullable', 'mimes:jpg,jpeg,png', 'max:1024'],
            ])->validateWithBag('updateProfileInformation');

            if ($input['email'] != null) {
                Validator::make($input, [
                    'email' => ['required', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
                ]);
            }
            if ($input['about_me'] != null) {
                Validator::make($input, [
                    'about_me' => ['nullable', 'string', 'max:255'],
                ]);
            }

            if (
                $input['email'] !== $user->email &&
                $user instanceof MustVerifyEmail
            ) {
                $this->updateVerifiedUser($user, $input);
            } else {
                $user->forceFill([
                    'name' => $input['name'],
                    'email' => $input['email'],
                    'about_me' => $input['about_me'],
                    'profile_photo_path' => fileUpload($input['photo'], UPLOAD_FOLDER_USER, $user->getAttributes()['profile_photo_path'])[0],
                ])->save();
            }
            $notification = array(
                'message' => $user->name . '\'s personal information has been updated successfully',
                'alert-type' => 'success'
            );
            return back()->with($notification);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    protected function updateVerifiedUser($user, array $input)
    {
        try {
            $user->forceFill([
                'name' => $input['name'],
                'phone' => $input['phone'],
                'email' => $input['email'],
                'email_verified_at' => null,
                'profile_photo_path' => fileUpload($input['photo'], UPLOAD_FOLDER_USER, $user->getAttributes()['profile_photo_path'])[0],
            ])->save();


            $notification = array(
                'message' => $user->name . '\'s personal information has been updated successfully',
                'alert-type' => 'success'
            );
            return back()->with($notification);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function adminDelete(Request $request)
    {
        try {
            $user = Auth::user();
            if (!isset($request->password) || !Hash::check($request->password, $user->password)) {
                $notification = array(
                    'message' => 'The provided password does not match your current password.',
                    'alert-type' => 'error'
                );
                return back()->with($notification);
            }
            $user->deleteProfilePhoto();
            $user->tokens->each->delete();
            $user->delete();
            return redirect()->back();
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }


    public function getContactMessage()
    {
        try {
            $title = 'Contact Us';
            if (request()->ajax()) {
                $get = Contact::all();
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        $button  = "<div class='d-flex'>
                                <a href='" . route('admin.contact.view', $get->id) . "'>
                                    <button type='button' class='btn btn-sm btn btn-success m-1' data-id='" . $get->id . "'>" . __('View') . "</button>
                                </a>
                                <a href='javascript:void(0)' title='Delete' class='sliderDestroyBtn'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.contact.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                        return $button;
                    })
                    ->addColumn('status', function ($get) {
                        return $get->status == 1 ? "Unread" : "Read";
                    })
                    ->addColumn('name', function ($get) {
                        return $get->f_name . ' ' . $get->l_name;
                    })
                    ->rawColumns(['action', 'status', 'name'])->make(true);
            }
            return view('backend.pages.contacts.index', compact('title'));
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function viewContactMessage($id)
    {
        try {
            $title = 'Contact';
            $contact = Contact::findOrfail($id);
            $contact->status = 2;
            $contact->save();
            return view('backend.pages.contacts.show', compact('contact', 'title'));
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function destroyContactMessage($id)
    {
        try {
            $contact = Contact::findOrfail($id);
            $contact->delete();
            return $this->backWithSuccess('Contact deleted successfully.');
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }


    public function newsLetter()
    {
        try {
            if (request()->ajax()) {
                $get = NewsLatter::query();
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        return "<div class='d-flex'>
                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.newsletter.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                    })
                    ->rawColumns(['action'])->make(true);
            }

            $title = 'News Letters';
            return view('backend.pages.news-letter.index', compact('title'));
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function newsLetterDestroy($id)
    {
        try {
            $contact = NewsLatter::findOrfail($id);
            $contact->delete();
            return $this->backWithSuccess('Deleted successfully.');
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
}
