<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\HotelService;
use App\Services\PackageService;
use App\Services\DiscountSubService;
use App\Services\DiscountService;
use Yajra\DataTables\Facades\DataTables;

class AdminDiscountSubController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    protected $discountSub;
    protected $package;
    protected $hotel;
    protected $discount;
    public function __construct()
    {
        parent::__construct();
        $this->discountSub = new DiscountSubService;
        $this->package = new PackageService;
        $this->hotel = new HotelService;
        $this->discount = new DiscountService;
    }
    public function index(Request $request)
    {
        try {
            if (request()->ajax()) {
                $get = $this->discountSub->getData();
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        return "<div class='d-flex'>
                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.discount-group.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                    })
                    ->filter(function ($instance) use ($request) {
                        if ($request->get('discount_id')) {
                            $instance->where('discount_id', $request->get('discount_id'));
                        }
                        if (!empty($request->get('search'))) {
                            $instance->whereHas('discount', function ($w) use ($request) {
                                $search = $request->get('search');
                                $w->where('title', 'LIKE', "%$search%");
                            });
                        }
                    })
                    ->addColumn('discount', function ($get) {
                        return ($get->discount ? $get->discount->title : '');
                    })
                    ->addColumn('discount_type', function ($get) {
                        return ($get->discount ? discountType($get->discount->discount_type) : '');
                    })
                    ->addColumn('amount', function ($get) {
                        return ($get->discount_type == DISCOUNT_FIXED ? currencyFormat($get->amount) : $get->amount . '%');
                    })
                    ->addColumn('related', function ($get) {
                        if ($get->hotel) {
                            return __('Hotel:') . $get->hotel->hotel_name;
                        }
                        if ($get->package) {
                            return __('Package:') . $get->package->title;
                        }
                    })
                    ->rawColumns(['action', 'discount', 'discount_type', 'amount', 'related'])
                    ->make(true);
            }
            return view('backend.pages.discountsub.table', [
                'title' => 'Discount Group',
                'discount' => $this->discount->getData()->get(),
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        try {
            return view('backend.pages.discountsub.form', [
                'title' => 'Discount Group',
                'package' => $this->package->getData()->get(),
                'hotel' => $this->hotel->getData()->get(),
                'discounts' => $this->discount->getData()->get(),
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'package.*' => 'required|string|max:255',
            'hotel.*' => 'required|string|max:255',
            'ammount' => 'required|string|max:255|regex:/^([0-9.]+)$/',
            'start_date' => 'required|string|max:255',
            'end_date' => 'required|string|max:255',
        ]);
        try {

            $data = $this->discountSub->store([
                'package_id' => $request->package,
                'hotel_id' => $request->hotel,
                'start_date_time' => $request->start_date,
                'ammount' => $request->ammount,
                'end_date_time' => $request->end_date,
            ]);
            return $this->backWithSuccess('Discount Sub has been created successfully');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $sub = $this->discountSub->getData()->where(['discount_id' => $id])->get();
        return response()->json($sub);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        try {
            return view('backend.pages.discountsub.edit', [
                'title' => 'Discount Group',
                'discountsub' => $this->discountSub->findById($id),
                'package' => $this->package->getData()->get(),
                'hotel' => $this->hotel->getData()->get(),
                'discounts' => $this->discount->getData()->get(),
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'package' => 'required|string|max:255',
            'hotel' => 'required|string|max:255',
            'discount' => 'required|string|max:255',
            'ammount' => 'required|string|max:255|regex:/^([0-9.]+)$/',
            'start_date' => 'required|string|max:255',
            'end_date' => 'required|string|max:255',
        ]);
        try {
            $this->discountSub->update([
                'package_id' => $request->package,
                'hotel_id' => $request->hotel,
                'discount_id' => $request->discount,
                'start_date_time' => $request->start_date,
                'ammount' => $request->ammount,
                'end_date_time' => $request->end_date,
            ], $id);
            return $this->backWithSuccess('Discount Sub has been Updated successfully');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $this->discountSub->destroy($id);
            return $this->backWithSuccess('Discount Sub Deleted successfully.');
        } catch (\Throwable $th) {
            return $this->backWithError($th->getMessage());
        }
    }
}
