<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use App\Services\EventServices;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class AdminEventController extends Controller
{
    protected $event;
    public function __construct()
    {
        Parent::__construct();
        $this->event = new EventServices();
    }

    public function index()
    {
        try {
            if (request()->ajax()) {
                $get = $this->event->getData()->get();
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        $button  = "<div class='d-flex'>
                                <a href='" . route('admin.event.edit', $get->id) . "'>
                                    <button type='button' class='btn btn-sm btn btn-success m-1 blogCategoryEditBtn' data-id='" . $get->id . "'>" . __('Edit') . "</button>
                                </a>

                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.event.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                        return $button;
                    })
                    ->addColumn('image', function ($get) {
                        return imageShow($get->image, 'Image', '', 'width="80"');
                    })
                    ->addColumn('date_time', function ($get) {
                        return date('Y-m-d h:i A', strtotime($get->date . ' ' . $get->time));
                    })
                    ->rawColumns(['action', 'image', 'date_time'])->make(true);
            }
            return view('backend.pages.event.index', [
                'title' => 'Events'
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
    public function create()
    {
        try {
            return view('backend.pages.event.form', [
                'title' => 'Add New Event',
                'service' => null
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required',
            'date'=> 'required',
            'image.*' => ['nullable', 'mimes:jpeg,png,jpg,gif']
        ], [
            'title' => 'Event title is required.',
        ]);

        try {
            $data = [
                'title' => $request->title,
                'description' => $request->description,
                'location' => $request->location,
                'date' => $request->date,
                'time' => $request->time
            ];
            if ($request->hasFile('image')) {
                $data['image'] = fileUpload($request->image, UPLOAD_FOLDER_EVENTS)[0];
            }

            $this->event->store($data);
            return $this->backWithSuccess('Event created successfully.');
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function edit($id)
    {
        try {
            $service = $this->event->findById($id);
            return view('backend.pages.event.form', [
                'title' => 'Edit Event',
                'service' => $service
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
    public function update(Request $request, $id)
    {
        try {
            $request->validate([
                'title' => 'required|string|max:255',
                'description' => 'required',
                'date'=> 'required',
                'image.*' => ['nullable', 'mimes:jpeg,png,jpg,gif']
            ], [
                'title' => 'Event title is required.',
            ]);
            $service = $this->event->findById($id);

            $data = [
                'title' => $request->title,
                'description' => $request->description,
                'location' => $request->location,
                'date' => $request->date,
                'time' => $request->time
            ];
            if ($request->hasFile('image')) {
                $data['image'] = fileUpload($request->image, UPLOAD_FOLDER_EVENTS, $service->getAttributes()['image'])[0];
            }
            $service->update($data);

            return $this->backWithSuccess('Event created successfully.');
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    public function destroy($id)
    {
        try {
            $service = $this->event->findById($id);
            destroryFile(UPLOAD_FOLDER_EVENTS, $service->getAttributes()['image']);
            $service->delete();
            return $this->backWithSuccess('Event has been deleted successfully.');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
}
