<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use App\Models\Country;
use App\Services\HotelService;
use App\Services\AccommodationService;
use App\Services\CityService;
use App\Services\FacilityService;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Yajra\DataTables\Facades\DataTables;

class AdminHotelsController extends Controller
{
    protected $hotel;
    protected $facility;
    protected $accommodation;
    protected $cities;
    public function __construct()
    {
        Parent::__construct();
        $this->hotel = new HotelService();
        $this->accommodation = new AccommodationService();
        $this->facility = new FacilityService();
        $this->cities = new CityService();
    }
    public function index(Request $request)
    {
        try {
            if ($request->ajax() && $request->id) {
                $cities = $this->cities->getData()->where(['country_id' => $request->id])->get();
                $data = '';
                foreach ($cities as $city) {
                    $data .= '<option value="' . $city->id . '">' . $city->name . '</option>';
                }
                return response()->json(['data' => $data]);
            }

            if (request()->ajax()) {
                $get = $this->hotel->getData();
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        return "<div class='d-flex'>
                                <a href='" . route('admin.hotel.edit', $get->id) . "'>
                                    <button type='button' class='btn btn-sm btn btn-success m-1 blogCategoryEditBtn' data-id='" . $get->id . "'>" . __('Edit') . "</button>
                                </a>

                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.hotel.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                    })
                    ->addColumn('description', function ($get) {
                        return Str::limit($get->description, 20);
                    })
                    ->addColumn('hotel_location', function ($get) {
                        $loaction = $get->hotel_location;
                        if ($get->hotelCountry) $loaction .= '<br /><b>' . __('Country: ') . '</b>' . $get->hotelCountry->name;
                        if ($get->hotelCity) $loaction .= '<br /><b>' . __('City: ') . '</b>' . $get->hotelCity->name;
                        return $loaction;
                    })
                    ->addColumn('hotel_primary_image', function ($get) {
                        return imageShow($get->hotel_primary_image, 'Hotel Image', '', "width='100' height='70'");
                    })
                    ->addColumn('status', function ($get) {
                        return "<label class='switch'>
                            <input type='checkbox' " . ($get->status == ACTIVE_STATUS ? 'checked' : '') . " id='" . $get->id . "' class='serviceActivationBtn'>
                            <span class='slider round'></span>
                        </label>";
                    })
                    ->rawColumns(['action', 'description', 'hotel_location', 'hotel_primary_image', 'status'])->make(true);
            }

            return view('backend.pages.hotel.index', [
                'title' => 'Hotel'
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function create()
    {
        try {
            return view('backend.pages.hotel.form', [
                'title' => 'Hotel',
                'hotel' => null,
                'cities' => [],
                'facilities' => $this->facility->getData()->get(),
                'accommodations' => $this->accommodation->getData()->get(),
                'hotel_facilities' => [],
                'countries' => Country::all()
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function show($id)
    {
        $item = $this->hotel->findById($id);
        try {
            if ($item->status) {
                $item->status = false;
                $item->save();
                $msg = 'Hotel deactivation successfully.';
                $status = true;
            } else {
                $item->status = true;
                $item->save();
                $msg = 'Hotel activation successfully.';
                $status = false;
            }
            return response()->json(['msg' => $msg, 'status' => $status]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function store(Request $request)
    {
        $request->validate([
            'country' => 'required|exists:countries,id',
            'city' => 'required|exists:cities,id',
            'hotel_name' => 'required|string|max:255',
            'hotel_location' => 'required',
            'description' => 'required',
            'latitude' => ['required', 'min:-90', 'max:90'],
            'longitude' => ['required', 'min:-180', 'max:180'],
            'breakfast_info' => 'required',
            'phone' => ['required'],
            'email' => ['required', 'email'],
            'facilities' => ['required', 'array', 'min:1'],
            'hotel_gd_image.*' => ['nullable', 'mimes:jpeg,png,jpg,gif', 'min:1'],
            'feature_image.*' => ['nullable', 'mimes:jpeg,png,jpg,gif', 'min:1'],
        ], [
            'hotel_name' => 'Hotel name is required.',
            'facilities.required' => 'Facilities is required.',
        ]);
        try {
            $hotel = [];

            if ($request->hasFile('hotel_gd_image')) {
                $hotel['hotel_primary_image'] = fileUpload($request->hotel_gd_image, '/upload/hotels')[0];
            }
            if ($request->hasFile('feature_image')) {
                $hotel['feature_image'] = fileUpload($request->feature_image, '/upload/hotels')[0];
            }

            $hotel['hotel_name']  =  $request->hotel_name;
            $hotel['hotel_location']  =  $request->hotel_location;
            $hotel['description']  =  $request->description;
            $hotel['rules']  =  $request->rules;
            $hotel['latitude']  =  $request->latitude;
            $hotel['longitude']  =  $request->longitude;
            $hotel['country']  =  $request->country;
            $hotel['city']  =  $request->city;
            $hotel['phone']  =  $request->phone;
            $hotel['email']  =  $request->email;
            $hotel['website']  =  $request->website;
            $hotel['breakfast_info']  =  $request->breakfast_info;
            $hotel['lunch_info']  =  $request->lunch_info;
            $hotel['dinner_info']  =  $request->dinner_info;
            $hotel['parking_info']  =  ($request->parking_info ? ACTIVE_STATUS : DEACTIVE_STATUS);
            $hotel['hotel_to_stay_title']  =  $request->hotel_to_stay_title;
            $hotel['hotel_to_stay_description']  =  $request->hotel_to_stay_description;
            $hotel['facilities']  =  json_encode($request->facilities, true);
            $hotel['status'] = ($request->status ? ACTIVE_STATUS : DEACTIVE_STATUS);

            $hotel = $this->hotel->store($hotel);

            return $this->backWithSuccess('Hotel created successfully.', 'admin.hotel-accommodation.add', $hotel->id);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
    public function edit($id)
    {
        try {
            $hotel = $this->hotel->findById($id);
            $cities = $this->cities->getData()->where(['country_id' => $hotel->country])->get();
            return view('backend.pages.hotel.form', [
                'title' => 'Hotel Edit',
                'hotel' => $hotel,
                'cities' => $cities,
                'countries' => Country::all(),
                'facilities' => $this->facility->getData()->get(),
                'accommodations' => $this->accommodation->getData()->get(),
                'hotel_facilities' => json_decode($hotel->facilities, true)
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
    public function update(Request $request, $id)
    {
        $request->validate([
            'country' => 'required|exists:countries,id',
            'city' => 'required|exists:cities,id',
            'hotel_name' => 'required|string|max:255',
            'hotel_location' => 'required',
            'description' => 'required',
            'latitude' => ['required', 'min:-90', 'max:90'],
            'longitude' => ['required', 'min:-180', 'max:180'],
            'breakfast_info' => 'required',
            'phone' => ['required'],
            'email' => ['required', 'email'],
            'facilities' => ['required', 'array', 'min:1'],
            'hotel_gd_image.*' => ['nullable', 'mimes:jpeg,png,jpg,gif', 'min:1'],
            'feature_image.*' => ['nullable', 'mimes:jpeg,png,jpg,gif', 'min:1'],
        ], [
            'hotel_name' => 'Hotel name is required.',
            'facilities.required' => 'Facilities is required.',
        ]);
        try {
            $hotel = $this->hotel->findById($id);

            if ($request->hasFile('hotel_gd_image')) {
                $hotel->hotel_primary_image = fileUpload($request->hotel_gd_image, UPLOAD_FOLDER_HOTELS, $hotel->getAttributes()['hotel_primary_image'])[0];
            }
            if ($request->hasFile('feature_image')) {
                $hotel->feature_image = fileUpload($request->feature_image, UPLOAD_FOLDER_HOTELS, $hotel->getAttributes()['feature_image'])[0];
            }
            $hotel->hotel_name  =  $request->hotel_name;
            $hotel->hotel_location  =  $request->hotel_location;
            $hotel->description  =  $request->description;
            $hotel->rules  =  $request->rules;
            $hotel->latitude  =  $request->latitude;
            $hotel->longitude  =  $request->longitude;
            $hotel->country =  $request->country;
            $hotel->city  =  $request->city;
            $hotel->phone  =  $request->phone;
            $hotel->email  =  $request->email;
            $hotel->website  =  $request->website;
            $hotel->breakfast_info  =  $request->breakfast_info;
            $hotel->lunch_info  =  $request->lunch_info;
            $hotel->dinner_info  =  $request->dinner_info;
            $hotel->parking_info  =  ($request->parking_info ? ACTIVE_STATUS : DEACTIVE_STATUS);
            $hotel->hotel_to_stay_title  =  $request->hotel_to_stay_title;
            $hotel->hotel_to_stay_description  =  $request->hotel_to_stay_description;
            $hotel->facilities  =  json_encode($request->facilities, true);
            $hotel['status'] = ($request->status ? ACTIVE_STATUS : DEACTIVE_STATUS);
            $hotel->save();
            return $this->backWithSuccess('Hotel Updated successfully.', 'admin.hotel-accommodation.add', $id);
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    public function destroy($id)
    {
        try {
            $service = $this->hotel->destroy($id);

            return $this->backWithSuccess('Hotel has been deleted successfully.');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
}
