<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\PackageDepartureService;
use App\Services\PackageService;
use Yajra\DataTables\Facades\DataTables;

class AdminPackageDepartureController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    protected $package;
    protected $package_departure;
    public function __construct()
    {
        Parent::__construct();
        $this->package = new PackageService();
        $this->package_departure = new PackageDepartureService;
    }
    public function index(Request $request)
    {
        try {
            if (request()->ajax()) {
                $get = $this->package_departure->getData();
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        return "<div class='d-flex'>
                                    <a href='" . route('admin.package-departure.edit', $get->id) . "'>
                                    <button type='button' class='btn btn-sm btn btn-success m-1 blogCategoryEditBtn' data-id='" . $get->id . "'>" . __('Edit') . "</button>
                                </a>
                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.package-departure.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                    })
                    ->addColumn('package', function ($get) {
                        return (($get->package) ? $get->package->title : '');
                    })
                    ->addColumn('status', function ($get) {
                        return statusActiveDeactive($get->booking_status);
                    })
                    ->filter(function ($get) use ($request) {
                        if ($request->get('package_id')) {
                            $get->where('package_id', $request->get('package_id'));
                        }
                        if (!empty($request->get('search'))) {
                            $get->whereHas('package', function ($w) use ($request) {
                                $search = $request->get('search');
                                $w->where('title', 'LIKE', "%$search%")
                                ->orWhere('sub_title', 'LIKE', "%$search%");
                            });
                        }
                    })
                    ->rawColumns(['action', 'package', 'status'])
                    ->make(true);
            }
            
            return view('backend.pages.packages.departure.table', [
                'title' => 'Pakage Departure',
                'departures' => $this->package_departure->getData()->get(),
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        try {
            return view('backend.pages.packages.departure.form', [
                'title' => 'Pakage Departure',
                'packages' => $this->package->getData()->get(),
                'package' => null,
                'departure' => null
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function packageDepartureAdd($package_id)
    {
        try {
            $package = $this->package->findById($package_id);
            $departure = $package->departure;
            return view('backend.pages.packages.departure.form', [
                'title' => 'Pakage Departure',
                'package' => $this->package->findById($package_id),
                'packages' => [],
                'departure' => $departure
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'package' => 'required|string|max:255',
            'departure_date' => 'required|string|max:255',
            'total_place' => 'required|string|max:255',
            'available_place' => 'required|string|max:255',
        ]);
        try {
            $data = [
                'package_id' => $request->package,
                'departure_date' => $request->departure_date,
                'total_place' => $request->total_place,
                'available_place' => $request->available_place,
                'booking_status' => (($request->booking_status) ? ACTIVE_STATUS : DEACTIVE_STATUS),
            ];
            if (empty($request->departure_id)) {
                $this->package_departure->store($data);
            } else {
                $this->package_departure->update($data, $request->departure_id);
            }

            return $this->backWithSuccess('Package Departure has been Stored successfully','admin.packages.index');
        } catch (\Illuminate\Validation\ValidationException $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        try {
            return view('backend.pages.packages.departure.edit', [
                'title' => 'Pakage Departure',
                'packages' => $this->package->getData()->get(),
                'departure' => $this->package_departure->findById($id),
            ]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'package' => 'required|string|max:255',
            'departure_date' => 'required|string|max:255',
            'total_place' => 'required|string|max:255',
            'available_place' => 'required|string|max:255',
            'booking_status' => 'nullable|string|max:20',
        ]);
        try {
            $data = [
                'package_id' => $request->package,
                'departure_date' => $request->departure_date,
                'total_place' => $request->total_place,
                'available_place' => $request->available_place,
                'booking_status' => ($request->booking_status == "on" ? 1 : 2),
            ];
            $this->package_departure->update($data, $id);
            return $this->backWithSuccess('Package Departure has been Updated successfully');
        } catch (\Illuminate\Validation\ValidationException $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $this->package_departure->destroy($id);
            return $this->backWithSuccess("Package Departure Deleted Successful");
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
}
