<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use App\Services\SliderServices;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class AdminSliderController extends Controller
{
    protected $slider;
    public function __construct()
    {
        Parent::__construct();
        $this->slider = new SliderServices();
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        try {
            if (request()->ajax()) {
                $get = $this->slider->getData()->get();
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        $button  = "<div class='d-flex'>
                                <a href='" . route('admin.slider.edit', $get->id) . "'>
                                    <button type='button' class='btn btn-sm btn btn-success m-1 blogCategoryEditBtn' data-id='" . $get->id . "'>" . __('Edit') . "</button>
                                </a>

                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.slider.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                        return $button;
                    })
                    ->addColumn('bg_image', function ($get) {
                        return imageShow($get->bg_image, 'Image', '', 'width="80"');
                    })
                    ->addColumn('status', function ($get) {
                        $status  = "<label class='switch'>
                <input type='checkbox' " . ($get->status ? 'checked' : '') . " id='" . $get->id . "' class='sliderActivationBtn'>
                <span class='slider round'></span>
            </label>";
                        return $status;
                    })
                    ->rawColumns(['action', 'bg_image', 'status'])->make(true);
            }
            return view('backend.pages.sliders.index', [
                'title' => 'Slider',
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        try {
            return view('backend.pages.sliders.form', [
                'title' => 'Add New Slider',
                'slider' => null
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'sub_title' => 'required|string|max:255',
            'button_name' => 'required|string|max:255',
            'button_url' => 'required|string|max:255',
            'description' => 'required',
            'bg_image.*' => ['required', 'mimes:jpeg,png,jpg,gif'],
        ], [
            'title' => 'Slider title is required.',
        ]);

        try {
            $slider = [];

            if ($request->hasFile('bg_image')) {
                $slider['bg_image'] = fileUpload($request->bg_image, UPLOAD_FOLDER_SLIDERS)[0];
            }
            $url = \Mews\Purifier\Facades\Purifier::clean($request->button_url, function (HTMLPurifier_Config $config) {
                $uri = $config->getDefinition('URI');
                $uri->addFilter(new HTMLPurifier_URIFilter_NameOfFilter(), $config);
            });

            $slider['title']       =  $request->title;
            $slider['sub_title']    =  $request->sub_title;
            $slider['button_name'] =  $request->button_name;
            $slider['button_url']  =  $url;
            $slider['description']  =  $request->description;
            $slider['status']      =  $request->status == 'on' ? true : false;

            $this->slider->store($slider);

            return $this->backWithSuccess('Slider created successfully.');
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $slider = $this->slider->findById($id);
        try {
            if ($slider->status) {
                $slider->status = false;
                $slider->save();
                $msg = 'Slider deactivation successfully.';
                $status = true;
            } else {
                $slider->status = true;
                $slider->save();
                $msg = 'Slider activation successfully.';
                $status = false;
            }
            return response()->json(['msg' => $msg, 'status' => $status]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        try {
            $slider = $this->slider->findById($id);
            return view('backend.pages.sliders.form', [
                'title' => 'Edit Slider',
                'slider' => $slider
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'sub_title' => 'required|string|max:255',
            'button_name' => 'required|string|max:255',
            'button_url' => 'required|string|max:255',
            'description' => 'required',
            'bg_image.*' => ['nullable', 'mimes:jpeg,png,jpg,gif'],
        ], [
            'title' => 'Slider title is required.',
        ]);

        try {

            $slider = $this->slider->findById($id);
            if ($request->hasFile('bg_image')) {
                $slider->bg_image = fileUpload($request->bg_image, UPLOAD_FOLDER_SLIDERS, $slider->getAttributes()['bg_image'])[0];
            }

            $slider->title       =  $request->title;
            $slider->sub_title    =  $request->sub_title;
            $slider->button_name =  $request->button_name;
            $slider->button_url  =  $request->button_url;
            $slider->description  =  $request->description;
            $slider->status      =  $request->status == 'on' ? true : false;

            $slider->save();

            return $this->backWithSuccess('Slider updated successfully.');
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $this->slider->destroy($id);
            return $this->backWithSuccess('has been deleted successfully.');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
}
