<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;

use App\Services\TeamServices;
use App\Services\DesignationService;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Yajra\DataTables\Facades\DataTables;

class AdminTeamController extends Controller
{
    protected $team;
    protected $designation;
    public function __construct()
    {
        Parent::__construct();
        $this->team = new TeamServices();
        $this->designation = new DesignationService();
    }
    /**
     * Display a listing of the resource.
     *
     */

    public function index()
    {
        try {
            if (request()->ajax()) {
                $get = $this->team->getData()->get();
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        $button  = "<div class='d-flex'>
                                <a href='" . route('admin.team.edit', $get->id) . "'>
                                    <button type='button' class='btn btn-sm btn btn-success m-1 blogCategoryEditBtn' data-id='" . $get->id . "'>" . __('Edit') . "</button>
                                </a>

                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('admin.team.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                        return $button;
                    })
                    ->addColumn('image', function ($get) {
                        return imageShow($get->image, 'Image', 'team-avatar-table', 'width="80"');
                    })
                    ->addColumn('designation', function ($get) {
                        return ($get->designation != null ? $get->designation->name : null);
                    })
                    ->addColumn('status', function ($get) {
                        $status  = "<label class='switch'>
                            <input type='checkbox' " . ($get->status ? 'checked' : '') . " id='" . $get->id . "' class='serviceActivationBtn'>
                            <span class='slider round'></span>
                        </label>";
                        return $status;
                    })
                    ->rawColumns(['image', 'designation', 'status','action'])->make(true);
            }

            return view('backend.pages.team.index', [
                'title' => 'Team',
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        try {
            $designation = $this->designation->getData()->get();
            return view('backend.pages.team.form', [
                'title' => 'Add New Member',
                'attorney' => null,
                'designation' => $designation
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'designation_id' => 'required',
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'phone' => ['required', 'string', 'max:20', 'regex:/^([0-9\s\-\+\(\)]*)$/'],
            'address' => ['required', 'string'],
            'description' => ['required', 'string'],
            'image.*' => ['nullable', 'mimes:jpeg,png,jpg,gif']
        ], [
            'designation_id.required' => 'Designation id is required.',
        ]);
        DB::beginTransaction();
        try {
            $attorney_user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'address' => $request->address,
                'password' => Hash::make('123456789'),
            ]);
            $attorney_user->assignRole('attorney');

            $attorney = [];
            if ($request->hasFile('image')) {
                $attorney['image'] = fileUpload($request->image, UPLOAD_FOLDER_ATTORNEYS)[0];
            }
            $attorney['name'] = $request->name;
            $attorney['designation_id'] =  $request->designation_id;
            $attorney['phone'] =  $request->phone;
            $attorney['email'] =  $request->email;
            $attorney['address'] =  $request->address;
            $attorney['description'] =  $request->description;
            $attorney['status'] = $request->status == 'on' ? true : false;
            $this->team->store($attorney);
            DB::commit();
            return $this->backWithSuccess('Volunteer created successfully, & Volunteer login default password is 123456789');
        } catch (\Throwable $th) {
            DB::rollBack();
            return redirect()->back()->with('error', $th->getMessage());
        }
    }



    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */

    public function teamShow($id)
    {
        $item = $this->team->findById($id);
        try {
            if ($item->status) {
                $item->status = false;
                $item->save();
                $msg = 'Team deactivation successfully.';
                $status = true;
            } else {
                $item->status = true;
                $item->save();
                $msg = 'Team activation successfully.';
                $status = false;
            }
            return response()->json(['msg' => $msg, 'status' => $status]);
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        try {
            $attorney = $this->team->findById($id);
            $designation = $this->designation->getData()->get();
            return view('backend.pages.team.form', [
                'title' => 'Edit Attorney',
                'attorney' => $attorney,
                'designation' => $designation
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            $attorney = $this->team->findById($id);
            if ($request->hasFile('image')) {
                $attorney['image'] = fileUpload($request->image, UPLOAD_FOLDER_ATTORNEYS)[0];
            }
            $attorney->name = $request->name;
            $attorney->designation_id =  $request->designation_id;
            $attorney->phone =  $request->phone;
            $attorney->email =  $request->email;
            $attorney->address =  $request->address;
            $attorney->description =  $request->description;
            $attorney->status = $request->status == 'on' ? true : false;
            $attorney->save();
            return $this->backWithSuccess('Team member updated successfully.');
        } catch (\Exception $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $attorney = $this->team->destroy($id);
            return $this->backWithSuccess('Attorney has been deleted successfully.');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
}
