<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use App\Http\Requests\FooterRequest;
use App\Http\Requests\SocialMediaRequest;
use App\Http\Requests\TopHeaderRequest;
use App\Models\FooterSettings;
use App\Models\GeneralSettings;
use App\Models\HeaderFooterSettings;
use App\Models\HeaderSettings;
use App\Models\SocialMediaSettings;
use Illuminate\Http\Request;
use App\Models\SEOSettings;
use App\Models\LogoSettings;
use Mews\Purifier\Purifier;

class AppSettingsController extends Controller
{
    public function __construct()
    {
        Parent::__construct();
    }


    public function getGeneralSettings()
    {
        try {
            $title = 'General Settings';
            $generalSetting = GeneralSettings::first();
            return view('backend.pages.settings.general', compact('title', 'generalSetting'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function saveGeneralSettings(Request $request)
    {
        $request->validate([
            'site_name' => 'required',
            'site_tag_line' => 'required',
            'site_sub_tag_line' => 'nullable',
            'author_name' => 'required',
            'og_meta_title' => 'nullable',
            'og_meta_description' => 'nullable',
            'og_meta_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg',
        ], [
            'site_name' => 'Site name is required.',
            'site_tag_line' => 'Site tag line is required.',
            'author_name' => 'Author name  is required.',
        ]);

        try {
            $generalSettingStore = GeneralSettings::first() ? GeneralSettings::first() : new GeneralSettings();
            $generalSettingStore->site_name = $request->site_name;
            $generalSettingStore->site_tag_line = $request->site_tag_line;
            $generalSettingStore->site_sub_tag_line = $request->site_sub_tag_line;
            $generalSettingStore->author_name = $request->author_name;
            $generalSettingStore->og_meta_title = $request->og_meta_title;
            $generalSettingStore->og_meta_description = $request->og_meta_description;
            $generalSettingStore->currency_symbol = $request->currency_symbol;
            $generalSettingStore->google_map_api_key = $request->google_map_api_key;
            if ($request->hasFile('og_meta_image')) {
                $generalSettingStore->og_meta_image = fileUpload($request->og_meta_image, UPLOAD_FOLDER_SETTINGS, (($generalSettingStore && $generalSettingStore->og_meta_image) ? $generalSettingStore->getAttributes()['og_meta_image'] : ''))[0];
            }

            $generalSettingStore->save();
            return $this->backWithSuccess('General Settings updated successfully.');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function topHeaderIndex()
    {
        try {
            $title = 'Top Header';
            $headerSettingValue = HeaderSettings::first();
            return view('backend.pages.settings.top-header-settings', compact('title', 'headerSettingValue'));
        } catch (\Throwable $th) {
            return $this->backWithError($th->getMessage());
        }
    }

    public function topHeaderStore(TopHeaderRequest $request)
    {
        try {
            $headerSettingStore = HeaderSettings::first() ? HeaderSettings::first() : new HeaderSettings();
            $headerSettingStore->show = $request->has('show');
            $headerSettingStore->left_content = $request->left_content;
            $headerSettingStore->right_content = $request->right_content;
            $headerSettingStore->right_content_login_button = $request->has('right_content_login_button');
            $headerSettingStore->search_blog_show = $request->has('search_blog_show');
            $headerSettingStore->save();
            return $this->backWithSuccess('Header Settings created successfully.');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }


    public function footerIndex()
    {
        try {
            $title = 'Footer Settings';
            $footer = FooterSettings::first();
            return view('backend.pages.settings.footer-settings', compact('title', 'footer'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function storeFooter(FooterRequest $request)
    {
        try {
            $group = FooterSettings::first() ? FooterSettings::first() : new FooterSettings();
            $inputs = $request->all();
            unset($inputs['page']);
            unset($inputs['group']);
            unset($inputs['_token']);

            $inputs = (object)$inputs;
            if (!$request->has('show')) {
                $group->update(['show' => false]);
            }

            if (!$request->has('show_social')) {
                $group->update(['show_social' => false]);
            }


            foreach ($inputs as $key => $value) {
                if ($key == 'footer_logo') {
                    $group->footer_logo  = fileUpload($value, UPLOAD_FOLDER_SETTINGS, ($group ? $group->getAttributes()['footer_logo'] : ''))[0];
                } else {
                    if ($key == 'show') {
                        $group->update([$key => true]);
                    } elseif ($key == 'show_social') {
                        $group->update([$key => true]);
                    } else {
                        $group->update([$key => $value]);
                    }
                }
                $group->save();
            }

            return $this->backWithSuccess('Saved Successfully');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function getLogoFaviconSettings()
    {
        try {
            $title = 'Logo Settings';
            $logoSettings = LogoSettings::first();
            return view('backend.pages.settings.logo-favicon', compact('title', 'logoSettings'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function saveLogoFaviconSettings(Request $request)
    {
        $logoFaviconSetting = LogoSettings::first();

        $hasLogo = !empty($logoFaviconSetting->logo);
        $hasFavicon = !empty($logoFaviconSetting->favicon);

        $request->validate([
            'logo' => $hasLogo ? 'nullable|array|min:1' : 'required|array|min:1',
            'logo.*' => 'mimes:jpeg,png,jpg,webp|max:2048',

            'favicon' => $hasFavicon ? 'nullable|array|min:1' : 'required|array|min:1',
            'favicon.*' => 'mimes:jpeg,png,jpg,webp|max:2048',
        ]);


        try {
            $logoFaviconSettingStore = $logoFaviconSetting ?? new LogoSettings();

            if ($request->file('logo')) {
                $uploadedLogo = fileUpload(
                    $request->file('logo'),
                    UPLOAD_FOLDER_SETTINGS,
                    $logoFaviconSettingStore->logo ?? ''
                );
                $logoFaviconSettingStore->logo = $uploadedLogo[0] ?? null;
            }

            if ($request->file('favicon')) {
                $uploadedFavicon = fileUpload(
                    $request->file('favicon'),
                    UPLOAD_FOLDER_SETTINGS,
                    $logoFaviconSettingStore->favicon ?? ''
                );
                $logoFaviconSettingStore->favicon = $uploadedFavicon[0] ?? null;
            }

            $logoFaviconSettingStore->save();

            return $this->backWithSuccess('Logo sections are updated successfully.');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', 'Something went wrong! ' . $th->getMessage());
        }
    }


    public function getSeoSettings()
    {
        try {
            $title = 'Seo Settings';
            $seoSetting = SEOSettings::first();
            return view('backend.pages.settings.seo', compact('title', 'seoSetting'));
        } catch (\Throwable $th) {
            return $this->backWithError($th->getMessage());
        }
    }

    public function saveSeoSettings(Request $request)
    {
        $request->validate([
            'meta_keyword' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:255',
        ]);

        if (!$request->meta_keyword && !$request->meta_description) {
            return redirect()->back()->with('error', 'Either Meta Keywords or Meta Description is required.');
        }
        try {
            $seoSettingStore = SEOSettings::first() ? SEOSettings::first() : new SEOSettings();
            $seoSettingStore->meta_keyword = $request->meta_keyword;
            $seoSettingStore->meta_description = $request->meta_description;
            $seoSettingStore->save();
            return $this->backWithSuccess('SEO Settings created successfully.');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function getSmtpSettings()
    {
        try {
            $title = 'Smtp Settings';
            return view('backend.pages.settings.smtp', compact('title'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function saveSmtpSettings(Request $request)
    {
        $request->validate([
            'mail_driver' => 'required',
            'mail_host' => 'required',
            'mail_port' => 'required',
            'mail_username' => 'required',
            'mail_password' => 'required',
            'mail_encryption' => 'required',
            'mail_from_address' => 'required',
        ]);

        try {
            $path = base_path('.env');

            if (file_exists($path)) {
                envUpdate('MAIL_MAILER', $request->mail_driver);
                envUpdate('MAIL_HOST', $request->mail_host);
                envUpdate('MAIL_PORT', $request->mail_port);
                envUpdate('MAIL_USERNAME', $request->mail_username);
                envUpdate('MAIL_PASSWORD', $request->mail_password);
                envUpdate('MAIL_ENCRYPTION', $request->mail_encryption);
                envUpdate('MAIL_FROM_ADDRESS', $request->mail_from_address);
            }
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }

        return $this->backWithSuccess('Success');
    }


    public function getInsertHeaderFooterSettings()
    {
        try {
            $title = 'Insert Header Footer';
            $headerFooterData = HeaderFooterSettings::first();
            return view('backend.pages.settings.insert-header-footer', compact('title', 'headerFooterData'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function saveInsertHeaderFooterSettings(Request $request)
    {
        try {
            $headerFooterSettingStore = HeaderFooterSettings::first() ? HeaderFooterSettings::first() : new HeaderFooterSettings();
            $headerFooterSettingStore->header = $request->header;
            $headerFooterSettingStore->footer = $request->footer;
            $headerFooterSettingStore->save();
            return $this->backWithSuccess('Header Footer created successfully.');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }

        return $this->backWithSuccess('Success');
    }

    public function socialMediaSettings()
    {
        try {
            $title = 'Social Media Settings';
            $socialMediaSettingAll = SocialMediaSettings::all();
            return view('backend.pages.settings.social-media', compact('title', 'socialMediaSettingAll'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function saveSocialMediaSettings(SocialMediaRequest $request)
    {
        try {
            $inputs = $request->except('_token');
            foreach ($inputs as $key => $value) {
                if ($value != null) {
                    $value = \Mews\Purifier\Facades\Purifier::clean($value, function (HTMLPurifier_Config $config) {
                        $uri = $config->getDefinition('URI');
                        $uri->addFilter(new HTMLPurifier_URIFilter_NameOfFilter(), $config);
                    });
                }
                SocialMediaSettings::where('name', $key)->update(['url' => $value]);
            }
            return $this->backWithSuccess('Social Media Settings created successfully.');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
}
