<?php

namespace App\Http\Controllers\AdminControllers;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Yajra\DataTables\Facades\DataTables;

class UserController extends Controller
{
    public function __construct()
    {
        Parent::__construct();
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function roleIndex()
    {
        try {
            return view('backend.pages.users.roles.index', [
                'title' => 'User Roles',
                'roles' => Role::all()
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function roleCreate()
    {
        try {
            return view('backend.pages.users.roles.form', [
                'title' => 'Create New Role',
                'role' => null,
                'permissions' => Permission::all()
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function roleStore(Request $request)
    {
        $this->validate($request, [
            'name' => ['required', 'max:50']
        ], [
            'name.required' => 'This name field is required.',
            'name.max:50' => 'Role name can\'t more than 50 character.'
        ]);
        try {
            $role = Role::create([
                'name' => strtolower($request->name),
                'guard_name' => 'web'
            ]);
            $role->syncPermissions($request->permission);
            return $this->backWithSuccess('A new role created successfully');
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function roleEdit(Role $role)
    {
        if ($role->name == 'admin') {
            return $this->backWithWarning('Nobody can\'t edit Admin role.');
        }
        try {
            return view('backend.pages.users.roles.form', [
                'title' => 'Edit Role',
                'role' => $role,
                'permissions' => Permission::all()
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function getRolePermission(Role $role)
    {
        try {
            $permissions = $role->permissions;
            $data = [];
            foreach ($permissions as $permission) {
                $data[] = $permission->id;
            }
            return response()->json($data);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function roleUpdate(Request $request, Role $role)
    {
        $this->validate($request, [
            'name' => ['required', 'max:50']
        ], [
            'name.required' => 'This name field is required.',
            'name.max:50' => 'Role name can\'t more than 50 character.'
        ]);
        try {
            $role->name = strtolower($request->name);
            $role->save();
            $role->syncPermissions($request->permission);
            return $this->backWithSuccess('Role assigned with permission.');
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function roleDestroy(Role $role)
    {
        if ($role->name == 'admin') {
            return $this->backWithWarning('Nobody can\'t delete Admin role.');
        } elseif ($role->name == 'client') {
            return $this->backWithWarning('Nobody can\'t delete Client role.');
        }
        try {
            $users = User::role($role->name)->count();
            if ($users > 0) {
                return $this->backWithWarning('This Role already assigned with some users, please delete them and try again later...');
            }
            $role->syncPermissions([]);
            $role->delete();
            return $this->backWithSuccess('Role has been deleted successfully...');
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function clientIndex()
    {
        try {
            if (request()->ajax()) {
                $get = User::where(['user_type' => GUEST_USER]);
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        $button  = "<div class='d-flex'>
                                        <a href='javascript:void (0)' >
                                            <button type='button' class='btn btn-sm btn btn-success m-1 roleEditBtn' data-json='" . json_encode($get) . "'  data-toggle='modal' data-target='#exampleModal'>" . _('view') . "</button>
                                        </a>
                                    </div>";
                        return $button;
                    })
                    ->rawColumns(['action'])->make(true);
            }

            return view('backend.pages.users.client.index', [
                'title' => 'All Clients'
            ]);
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
}
