<?php

namespace App\Http\Controllers;

use App\Models\PageSectionSettings;
use App\Models\PageSettings;
use App\Models\User;
use App\Services\UserServices;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    protected $user;
    public function __construct()
    {
        parent::__construct();
        $this->user = new UserServices;
    }
    public function guestLoginPage()
    {
        try {
            if (Auth::check() && Auth::user()->user_type == GUEST_USER) {
                return redirect()->route('guest.dashboard');
            }
            $title = null;
            $pageContent = null;

            $page = PageSettings::where('name', PAGE_AUTH)->first();
            if (!empty($page)) {
                $pageContent = PageSectionSettings::where('name', PAGE_GROUP_AUTH_LOGIN)->first();
                if (!empty($pageContent)) {
                    $title = ucfirst($pageContent->title);
                } else {
                    $title = ucfirst($page->name);
                }
            }
            return view('frontend.auth.login', compact('title', 'pageContent'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    public function guestLoginAuthentication(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => ['required', 'email', 'exists:users,email'],
                'password' => ['required'],
            ]);

            if ($validator->fails()) {
                return redirect()->back()
                    ->withErrors($validator)
                    ->withInput();
            }

            if (Auth::attempt(['email' => $request->email, 'password' => $request->password, 'user_type' => GUEST_USER], $request->remember)) {
                $user = auth()->user();
                if (!$user->email_verified_at) {
                    $this->confirmMailSend($user);
                    Auth::logout();
                    return redirect()->back()->withErrors(['email' => __('Your email is not confirmed. A confirmation email was sent to your email account.')]);
                }
                return redirect()->route('guest.dashboard');
            }

            return $this->backWithError('The provided credentials do not match our records.')->withInput();
        } catch (\Throwable $th) {
            return $this->backWithError($th->getMessage());
        }
    }

    public function guestRegisterPage()
    {
        try {
            if (Auth::check() && Auth::user()->user_type == GUEST_USER) {
                return redirect()->route('guest.dashboard');
            }

            $title = null;
            $pageContent = null;

            $page = PageSettings::where('name', PAGE_AUTH)->first();
            if (!empty($page)) {
                $pageContent = PageSectionSettings::where('name', PAGE_GROUP_AUTH_REGISTER)->first();
                if (!empty($pageContent)) {
                    $title = ucfirst($pageContent->title);
                } else {
                    $title = ucfirst($page->name);
                }
            }
            return view('frontend.auth.register', compact('title', 'pageContent'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    public function guestRegisterStore(Request $request)
    {
        try {
            $massage = [];
            $user = User::whereEmail($request->email)->first();
            if ($user) {
                $msg = 'This email is already taken. ';
                if (!$user->email_verified_at) {
                    $this->confirmMailSend($user);
                    $msg .= 'And this email is not verified. A confirmation email was sent to this email account.';
                }
                $massage['email.unique'] = $msg;
            }

            $validator = Validator::make($request->all(), [
                'name' => ['required', 'string', 'max:255'],
                'email' => 'required|string|email|max:255|unique:users,email',
                'password' => ['required', 'confirmed'],
                'phone' => ['required'],
            ], $massage);

            if ($validator->fails()) {
                return redirect()->back()
                    ->withErrors($validator)
                    ->withInput();
            }

            // Start the transaction
            DB::beginTransaction();

            try {
                $data = [
                    'name' => $request->name,
                    'surname' => $request->surname,
                    'email' => $request->email,
                    'phone' => $request->phone,
                    'user_type' => GUEST_USER,
                    'password' => bcrypt($request->password)
                ];

                $user = $this->user->store($data);

                // Send confirmation email
                $this->confirmMailSend($user);

                // Commit the transaction
                DB::commit();

                return $this->backWithSuccess('Account created successfully', 'guest.login');
            } catch (\Throwable $th) {
                // Rollback the transaction in case of error
                DB::rollBack();
                return $this->backWithError($th->getMessage());
            }
        } catch (\Throwable $th) {
            return $this->backWithError($th->getMessage());
        }
    }

    private function confirmMailSend($user)
    {
        $token = base64_encode($user->email . '|' . $user->id);
        $data = [
            'to' => $user->email,
            'subject' => 'Email Confirmation',
            'message' => view('emails.html.confirm-email', compact('user', 'token'))->render()
        ];
        mailFunc($data);
    }

    public function guestForgetPassword()
    {
        $title = __('Forget Password');
        return view('frontend.auth.password-recovery', compact('title'));
    }

    public function guestResetPassword($token)
    {
        $title = __('Reset Password');
        $token = explode('|', base64_decode($token));
        $user_id = $token[1];

        $user = User::findorfail($user_id);
        if (!$user) {
            return $this->backWithError('Your link is broken', 'guest.forget.password');
        }
        $token_id = encrypt($user->id);
        return view('frontend.auth.reset-password', compact('token_id', 'title'));
    }

    public function guestPasswordReset(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'password' => ['required', 'confirmed', 'min:6'],
                'password_confirmation' => ['required', 'min:6'],
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator);
            }

            User::where(['id' => decrypt($request->_token_id)])->update(['password' => bcrypt($request->password)]);

            return $this->backWithSuccess('Your password update successfully', 'guest.login');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function guestEmailPassword(Request $request)
    {
        try {
            $request->validate(['email' => ['required', 'exists:users,email']]);

            $user = User::where('email', $request->email)->first();
            $token = base64_encode($user->email . '|' . $user->id);

            $data = [
                'to' => $user->email,
                'subject' => 'Reset password link',
                'message' => view('emails.html.reset-password-email', compact('user', 'token'))->render()
            ];

            mailFunc($data);

            return $this->backWithSuccess('A mail send to your email with password recovery link');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    public function guestEmailVerify($token)
    {
        try {
            $token = explode('|', base64_decode($token));
            $user_id = $token[1];

            $user = $this->user->findById($user_id);
            if (!$user) {
                return $this->backWithError('Your link is broken', 'guest.forget.password');
            }
            $user->update(['email_verified_at' => date('Y-m-d H:i:s')]);

            return $this->backWithSuccess('Account verified successfully', 'guest.login');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }



    public function adminLogin()
    {
        try {
            if (Auth::user()) {
                if (!Auth::user()->hasRole('client') && Auth::user()->user_type == ADMIN_USER) {
                    return redirect()->route('admin.dashboard');
                }
            }
            return view('auth.login');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function adminPostLogin(Request $request)
    {
        try {
            $request->validate([
                'email' => ['required', 'email'],
                'password' => ['required'],
            ]);

            if (Auth::attempt(['email' => $request->email, 'password' => $request->password, 'user_type' => ADMIN_USER], $request->remember)) {
                return redirect()->route('admin.dashboard');
            }
            return back()->withErrors([
                'email' => 'The provided credentials do not match our records.',
            ]);
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
}
