<?php

namespace App\Http\Controllers\GuestViewControllers;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Services\HotelBookingService;
use App\Services\PackageBookingService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Yajra\DataTables\Facades\DataTables;

class UserController extends Controller
{

    protected $hotel_booking;
    protected $package_booking;
    public function __construct()
    {
        parent::__construct();
        $this->hotel_booking = new HotelBookingService;
        $this->package_booking = new PackageBookingService;
    }

    public function guestLogout()
    {
        Auth::logout();

        return redirect()->route('guest.login');
    }
    public function guestDashboard(Request $request)
    {
        try {
            $title = __('Dashboard');
            $hotel_order_count = $this->hotel_booking->getData()->where(['user_id' => Auth::id()])->count();
            $package_order_count = $this->package_booking->getData()->where(['user_id' => Auth::id()])->count();
            return view('frontend.auth.pages.dashboard', compact('title', 'hotel_order_count', 'package_order_count'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    public function guestProfileUpdate(Request $request)
    {
        try {
            $title = __('Profile');
            return view('frontend.auth.pages.profile-update', compact('title'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    public function guestProfileUpdateStore(Request $request)
    {
        $request->validate([
            'name' => ['required'],
            'address' => ['required'],
        ]);
        try {
            $user = User::findorfail(Auth::id());
            $data = [
                'name' => $request->name,
                'address' => $request->address,
                'about_me' => $request->about_me
            ];
            $user->update($data);

            $notification = array(
                'message' => $user->name . '\'s personal information has been updated successfully',
                'alert-type' => 'success'
            );
            return back()->with($notification);
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    public function guestProfileImageUpdate(Request $request)
    {
        $request->validate([
            'profile_photo_path' => ['required', 'mimes:jpeg,jpg,png'],
        ]);
        try {
            $user = User::findorfail(Auth::id());
            if ($request->hasFile('profile_photo_path')) {
                $data['profile_photo_path'] = fileUpload($request->profile_photo_path, UPLOAD_FOLDER_USER, $user->getAttributes()['profile_photo_path'])[0];
            }
            $user->update($data);

            $notification = array(
                'message' => $user->name . '\'s image has been updated successfully',
                'alert-type' => 'success'
            );
            return back()->with($notification);
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }
    public function guestPasswordUpdate(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'current_password' => ['required'],
                'password' => ['required', 'confirmed', 'min:6'],
                'password_confirmation' => ['required', 'min:6'],
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator);
            }

            $user = User::findorfail(Auth::id());

            if (!Hash::check($request->current_password, $user->password)) {
                return $this->backWithError('Your current password input is wrong. Please try again');
            }

            $user->update([
                'password' => bcrypt($request->password)
            ]);

            Auth::login($user);

            return $this->backWithSuccess('Your password update successfully');
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function guestHotelOrder(Request $request)
    {
        try {
            if (request()->ajax()) {
                $get = $this->hotel_booking->getData()->where(['user_id' => Auth::id()]);
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        return "<div class='d-flex'>
                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('guest.hotel.order.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                    })
                    ->addColumn('hotel', function ($get) {
                        return ' <p class="mb-1">
                        <b>' . __('Name: ') . '</b>' . $get->hotel->hotel_name . '</p>
                        <p class="mb-1">
                            <b>' . __('Accommodation: ') . '</b>' . $get->hotelAccommodation->accommodation->name . '
                        </p>
                        <p class="mb-1">
                            <b>' . __('Accommodation Price: ') . '</b>' . currencyFormat($get->hotelAccommodation->price) . '
                        </p>';
                    })
                    ->addColumn('reservation', function ($get) {
                        return ' <p class="mb-1">
                        <b>' . __('Check In: ') . '</b>' . date('m-d-Y H:i A', strtotime($get->check_in)) . '</p>
                        <p class="mb-1">
                            <b>' . __('Check Out: ') . '</b>' . date('m-d-Y H:i A', strtotime($get->check_out)) . '
                        </p>
                        <p class="mb-1">
                            <b>' . __('Arrival Time: ') . '</b>' . date('m-d-Y H:i A', strtotime($get->arrival_date_time)) . '
                        </p>';
                    })
                    ->addColumn('status', function ($get) {
                        return bookignStatus($get->status);
                    })
                    ->filter(function ($instance) use ($request) {
                        if (!empty($request->get('search'))) {
                            $instance->whereHas('hotel', function ($w) use ($request) {
                                $search = $request->get('search');
                                $w->where('hotel_name', 'LIKE', "%$search%");
                            });
                        }
                    })
                    ->rawColumns(['action', 'hotel', 'reservation', 'status'])->make(true);
            }
            $title = __('Hotel Order');
            return view('frontend.auth.pages.hotel-order', compact('title'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function guestHotelOrderDelete($id)
    {
        try {
            $this->hotel_booking->destroy($id);
            return $this->backWithSuccess(__('Hotel booking delete successfully.'));
        } catch (\Throwable $th) {
            return $this->backWithError($th->getMessage());
        }
    }

    public function guestPackageOrder(Request $request)
    {
        try {
            if (request()->ajax()) {
                $get = $this->package_booking->getData()->where(['user_id' => Auth::id()]);
                return DataTables::of($get)
                    ->addIndexColumn()
                    ->addColumn('action', function ($get) {
                        return "<div class='d-flex'>
                                <a href='javascript:void(0)' title='Delete' class='deleteRow'>
                                    <button type='button' class='btn btn-sm btn btn-danger m-1'>" . __('Delete') . "</button>
                                    <form action=" . route('guest.package.order.destroy', $get->id) . " method='post' class='deleteForm'>
                                        " . csrf_field() . "
                                        <input type='hidden' name='_method' value='delete'>
                                    </form>
                                </a>
                            </div>";
                    })
                    ->addColumn('package', function ($get) {
                        return ' <p class="mb-1">
                        <b>' . __('Name: ') . '</b>' . $get->package->title . '</p>
                        <p class="mb-1">
                            <b>' . __('Group Person: ') . '</b>' . $get->packageSubPrice->group_size . '
                        </p>
                        <p class="mb-1">
                            <b>' . __('Group Price: ') . '</b>' . currencyFormat($get->packageSubPrice->tour_price) . '
                        </p>';
                    })
                    ->addColumn('arrival', function ($get) {
                        return '<p class="mb-1">
                        <b>' . __('In: ') . '</b>' . date('m-d-Y H:i A', strtotime($get->arrival_date_time)) . '</p>';
                    })
                    ->addColumn('status', function ($get) {
                        return bookignStatus($get->status);
                    })
                    ->filter(function ($instance) use ($request) {
                        if ($request->get('hotel_id')) {
                            $instance->where('hotel_id', $request->get('hotel_id'));
                        }
                        if (!empty($request->get('search'))) {
                            $instance->whereHas('hotel', function ($w) use ($request) {
                                $search = $request->get('search');
                                $w->where('hotel_name', 'LIKE', "%$search%");
                            });
                        }
                    })
                    ->rawColumns(['action', 'package', 'arrival', 'status'])->make(true);
            }
            $title = __('Package Order');
            return view('frontend.auth.pages.package-order', compact('title'));
        } catch (\Throwable $th) {
            return redirect()->back()->with('error', $th->getMessage());
        }
    }

    public function guestPackageOrderDelete($id)
    {
        try {
            $this->package_booking->destroy($id);
            return $this->backWithSuccess(__('Package booking delete successfully.'));
        } catch (\Throwable $th) {
            return $this->backWithError($th->getMessage());
        }
    }
}
